/*
 * File: FDICBankFailures.java
 * ==================================================================
 * A program to visualize the distribution and frequency of banks
 * closed by the Federal Deposit Insurance Corporation over the
 * years 2000 - 2012.
 */

import acm.program.*;
import acm.util.*;
import acm.graphics.*;
import java.util.*;
import java.io.*;
import java.awt.event.*;

public class FDICBankFailures extends Program {
	/* Our preferred size. */
	public static final int APPLICATION_WIDTH = 850;
	public static final int APPLICATION_HEIGHT = 650;
	
	/* The file in which all of the failed banking information is stored. */
	private static final String BANKING_FILE = "failed-banks.txt";

	/* The file containing the borders of the US. */
	private static final String BORDERS_FILE = "us-borders.txt";

	/* A map from dates to list of bank failures that occurred on that date. */
	private TreeMap<Date, List<BankFailure>> failures =
			new TreeMap<Date, List<BankFailure>>();

	/* The display. */
	private BankingDisplay display;

	/* Sliders controlling the start and end dates to visualize. */
	private DateSlider slider;

	public void init() {
		loadFailures();
		addDisplay();
		addInteractors();
	}

	/**
	 * Adds the control bars to the display.
	 */
	private void addInteractors() {
		/* Find the range of when failures occurred. */
		Date firstFailure = failures.firstKey();
		Date lastFailure = failures.lastKey();
		
		/* Add a slider to range over those dates. */
		slider = new DateSlider(firstFailure, lastFailure);
		slider.addActionListener(this);
		add(slider, SOUTH);
	}
	
	/**
	 * Updates the display as an action is performed.
	 */
	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == slider) {
			/* Find the start and end dates. */
			Date startDate = slider.getStartDate();
			Date endDate = slider.getEndDate();
			
			/* Determine the range of failures in that timespan. */
			Map<Date, List<BankFailure>> toDisplay =
					failures.subMap(startDate, endDate);
			
			/* Add them to the display. */
			display.clear();
			for (Date time: toDisplay.keySet()) {
				List<BankFailure> currFailures = toDisplay.get(time);
				for (BankFailure failure: currFailures) {
					display.addFailure(failure);
				}
			}
			display.update();
		}
	}

	/**
	 * Sets up and adds the display to the window.
	 */
	private void addDisplay() {
		display = new BankingDisplay(loadBorders());
		add(display);
	}

	/**
	 * Loads the list of bank failures from the data file.
	 */
	private void loadFailures() {
		try {
			BufferedReader br = new BufferedReader(new FileReader(BANKING_FILE));

			while (true) {
				String line = br.readLine();
				if (line == null)
					break;

				BankFailure failure = new BankFailure(line);

				/* Update the map by associating this time with the bank failure.
				 * If this is the first failure at this time, create a new spot in
				 * the map for it.
				 */
				if (!failures.containsKey(failure.getFailDate())) {
					failures.put(failure.getFailDate(), new ArrayList<BankFailure>());
				}

				failures.get(failure.getFailDate()).add(failure);
			}

			br.close();
		} catch (IOException e) {
			throw new ErrorException(e);
		}
	}

	/**
	 * Loads all of the borders of US states from the borders file, returning it
	 * as a map from state codes to the borders of those states.
	 * 
	 * @return A map from state codes to state borders.
	 */
	private Map<String, Border> loadBorders() {
		try {
			BufferedReader br = new BufferedReader(new FileReader(BORDERS_FILE));
			Map<String, Border> result = new HashMap<String, Border>();

			while (true) {
				/* Read in the name of the state and how many shapes there are. */
				String state = br.readLine();
				String numShapes = br.readLine();

				/* If we ran out of data, we're done. */
				if (numShapes == null) break;

				/* Fill in all of the shapes. */
				List<Shape> shapes = new ArrayList<Shape>();
				for (int i = 0; i < Integer.parseInt(numShapes); i++) {
					shapes.add(readShape(br));
				}

				/* Group these together into a Border. */
				result.put(state, new Border(shapes));
			}

			br.close();
			return result;
		} catch (IOException e) {
			throw new ErrorException(e);
		}
	}

	/**
	 * Reads a single shape from a file.
	 * 
	 * @param br The buffered reader containing the shape.
	 * @return The shape.
	 */
	private Shape readShape(BufferedReader br) {
		try {
			ArrayList<GPoint> points = new ArrayList<GPoint>();
			while (true) {
				String line = br.readLine();

				/* Stop if the file ended or if this was the end of the shape. */ 
				if (line == null || line.isEmpty()) break;

				/* Split the line into its longitude and latitude pieces. */
				String[] pieces = line.split(" ");

				/* Combine them into a GPoint. */
				GPoint pt = new GPoint(Double.parseDouble(pieces[0]), Double.parseDouble(pieces[1]));
				points.add(pt);
			}
			return new Shape(points);
		} catch (IOException e) {
			throw new ErrorException(e);
		}
	}
}
